const NUMBER_RANGE = [-7, 7];  // Updated to 7 units each side
const TOLERANCE = 15; // pixels

// Generate unique random numbers within range
function generateRandomNumbers(count) {
    const numbers = new Set();
    while(numbers.size < count) {
        numbers.add(Math.floor(Math.random() * 15) - 7); // -7 to 7
    }
    return Array.from(numbers);
}

let numbersToPlace = generateRandomNumbers(7); // Generate 7 random numbers
let placedNumbers = {};

function initNumberLine() {
    const container = document.getElementById('numberLine');
    container.innerHTML = '';
    
    // Create number line
    const line = document.createElement('div');
    line.className = 'number-line';
    container.appendChild(line);
    
    // Create ticks and labels
    for(let i = NUMBER_RANGE[0]; i <= NUMBER_RANGE[1]; i++) {
        const tick = document.createElement('div');
        tick.className = 'number-tick';
        tick.style.left = `${valueToPosition(i)}%`;
        container.appendChild(tick);
        
        if(i === 0) {
            const label = document.createElement('div');
            label.className = 'number-label';
            label.textContent = i;
            label.style.left = `${valueToPosition(i)}%`;
            container.appendChild(label);
        }
    }
}

function createDraggableNumbers() {
    const container = document.getElementById('numbersContainer');
    container.innerHTML = '';
    
    numbersToPlace.sort(() => Math.random() - 0.5).forEach(num => {
        const div = document.createElement('div');
        div.className = 'draggable-number';
        div.textContent = num;
        div.draggable = true;
        div.dataset.value = num;
        div.addEventListener('dragstart', handleDragStart);
        container.appendChild(div);
    });
}

function handleDragStart(e) {
    e.dataTransfer.setData('text/plain', e.target.dataset.value);
}

function handleDrop(e) {
    e.preventDefault();
    const value = parseInt(e.dataTransfer.getData('text/plain'));
    const container = document.getElementById('numberLine');
    const rect = container.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const expectedPosition = valueToPosition(value) * rect.width / 100;
    
    if(Math.abs(x - expectedPosition) < TOLERANCE && !placedNumbers[value]) {
        placeNumber(value, x - rect.width * 0.01);
        showFeedback('¡Correcto!');
        checkCompletion();
    }
}

function placeNumber(value, x) {
    const container = document.getElementById('numberLine');
    const div = document.createElement('div');
    div.className = 'placed-number draggable-number';
    div.textContent = value;
    div.style.position = 'absolute';
    div.style.left = `${x}px`;
    div.style.top = '50%';
    div.style.transform = 'translate(-50%, -50%)';
    container.appendChild(div);
    
    placedNumbers[value] = true;
    document.querySelector(`[data-value="${value}"]`).style.visibility = 'hidden';
}

function valueToPosition(value) {
    const range = NUMBER_RANGE[1] - NUMBER_RANGE[0];
    return ((value - NUMBER_RANGE[0]) / range) * 100;
}

function showFeedback(text) {
    const feedback = document.getElementById('feedback');
    feedback.textContent = text;
    feedback.style.animation = 'none';
    void feedback.offsetWidth;
    feedback.style.animation = 'pop 0.3s ease';
}

function checkCompletion() {
    if(Object.keys(placedNumbers).length === numbersToPlace.length) {
        showFeedback('¡Todos los números están correctos! ');
    }
}

function resetGame() {
    numbersToPlace = generateRandomNumbers(7); // Generate new random numbers
    placedNumbers = {};
    initNumberLine();
    createDraggableNumbers();
    document.getElementById('feedback').textContent = '';
}

function handleResize() {
    const container = document.getElementById('numberLine');
    const rect = container.getBoundingClientRect();
    
    document.querySelectorAll('.placed-number').forEach(num => {
        const value = parseInt(num.textContent);
        const newPosition = valueToPosition(value) * rect.width / 100;
        num.style.left = `${newPosition}px`;
    });
}

// Generate stars for space background
function createStars() {
    const container = document.body;
    const starCount = 100;
    
    for(let i = 0; i < starCount; i++) {
        const star = document.createElement('div');
        star.className = 'star';
        
        const size = Math.random() * 2 + 1;
        const x = Math.random() * 100;
        const y = Math.random() * 100;
        const duration = Math.random() * 1 + 0.5;
        
        star.style.cssText = `
            width: ${size}px;
            height: ${size}px;
            top: ${y}%;
            left: ${x}%;
            animation-duration: ${duration}s;
        `;
        
        container.appendChild(star);
    }
}

document.addEventListener('DOMContentLoaded', () => {
    createStars();
    initNumberLine();
    createDraggableNumbers();
    document.getElementById('numberLine').addEventListener('dragover', e => e.preventDefault());
    document.getElementById('numberLine').addEventListener('drop', handleDrop);
});

window.addEventListener('resize', () => {
    handleResize();
});